#include "pch.h"

#include "Common.hpp"
#include "Misc.hpp"

#include "LineFormat.h"

CLineFormat::CLineFormat(const tstring& aFormat) {
    setFormat(aFormat);
}


void CLineFormat::setFormat(const tstring& aFormat) {
    format = aFormat;
    parse();
}


void CLineFormat::parse() {

	int	sign;

    fields.clear();

	// we'll work with individual characters
	LPCTSTR	characters = format.c_str();

	CFormatField	field;

	int	fromPos    = 0;
	int	currentPos = 0;
	while (characters[currentPos] != 0) {
		if (characters[currentPos] == TEXT('%')) {
			// store data if it was defined
			if (currentPos > fromPos) {
				// tail presents; it is just a sequence of characters
				field.value     = tstring( characters + fromPos, currentPos-fromPos );
				field.type      = CFormatField::DATA;
				field.width     = 0;
				field.precision = 0;
				field.modificator = CFormatField::NONE;
				fields.push_back( field );
			}
			// reset values
			field.width       = 0;
			field.precision   = 0;
			field.modificator = CFormatField::NONE;
			// skip %
			currentPos++;
			// get width
			fromPos = currentPos;
			sign = 1;
			if (characters[currentPos] == TEXT('-')) {
				sign = -1;
				currentPos++;
				fromPos++;
			}
			while (isdigit( characters[currentPos] )) {
				currentPos++;
			}
			if (currentPos > fromPos) {
				// get width and store it
				LPTSTR	widthChars = new TCHAR[currentPos-fromPos + 1];
				memcpy( widthChars, characters + fromPos, (currentPos-fromPos)*sizeof(TCHAR) );
				widthChars[currentPos-fromPos] = 0;
				if (!StrToInt( widthChars, field.width )) {
					field.width = 0;
				}
				delete [] widthChars;
			}
			field.width *= sign;
			// check .
			if (characters[currentPos] == TEXT('.')) {
				// precision should be specified
				currentPos++;
				fromPos = currentPos;
				sign = 1;
				if (characters[currentPos] == TEXT('-')) {
					sign = -1;
					currentPos++;
					fromPos++;
				}
				while (isdigit( characters[currentPos] )) {
					currentPos++;
				}
				if (currentPos > fromPos) {
					// get precision and store it
					LPTSTR	precisionChars = new TCHAR[currentPos-fromPos + 1];
					memcpy( precisionChars, characters + fromPos, (currentPos-fromPos)*sizeof(TCHAR) );
					precisionChars[currentPos-fromPos] = 0;
					if (!StrToInt( precisionChars, field.precision )) {
						field.precision = 0;
					}
					delete [] precisionChars;
				}
				field.precision *= sign;
			}
			// check modificator
			switch (characters[currentPos]) {
			case TEXT('u'):
				field.modificator = CFormatField::UPPERCASE;
				currentPos++;
				break;
			case TEXT('l'):
				field.modificator = CFormatField::LOWERCASE;
				currentPos++;
				break;
			}
			// check type
			field.type = CFormatField::UNKNOWN;
			switch (characters[currentPos]) {
			case TEXT('n'):
				field.type = CFormatField::NAME;
				break;
			case TEXT('d'):
				field.type = CFormatField::DESCRIPTION;
				break;
			case TEXT('w'):
				field.type = CFormatField::MULTILINE_DESCRIPTION;
				break;
			case TEXT('s'):
				field.type = CFormatField::SIZE_BYTES;
				break;
			case TEXT('S'):
				field.type = CFormatField::SIZE_SEPARATED_BYTES;
				break;
			case TEXT('k'):
				field.type = CFormatField::SIZE_KBYTES;
				break;
			case TEXT('K'):
				field.type = CFormatField::SIZE_SEPARATED_KBYTES;
				break;
			case TEXT('m'):
				field.type = CFormatField::SIZE_MBYTES;
				break;
			case TEXT('M'):
				field.type = CFormatField::SIZE_SEPARATED_MBYTES;
				break;
			case TEXT('a'):
				field.type = CFormatField::AUTOSIZE;
				break;
			case TEXT('A'):
				field.type = CFormatField::SEPARATED_AUTOSIZE;
				break;
			}
			fields.push_back( field );
			if (field.type != CFormatField::UNKNOWN) {
				currentPos++;
			}
			fromPos = currentPos;
		}
		else {
			// go to the next symbol
			currentPos++;
		}
	}
	// store tail
	if (currentPos > fromPos) {
		// tail presents; it is just a sequence of characters
		field.value = tstring( characters + fromPos, currentPos-fromPos );
		field.type  = CFormatField::DATA;
		field.width       = 0;
		field.precision   = 0;
		field.modificator = CFormatField::NONE;
		fields.push_back( field );
	}
}


tstring CLineFormat::apply(const tstring& name, const tstring& desc, filesize_t size) {
    tstring result;
	LPCTSTR	valueData;

    // go through all fields
    vector<CFormatField>::iterator   it = fields.begin();
    while (it != fields.end()) {
        CFormatField&  field = (*it);

        tstring value;

        switch (field.type) {
		case CFormatField::UNKNOWN:
			value = TEXT("ERROR");
			break;
        case CFormatField::DATA:
            value = field.value;
            break;
        case CFormatField::NAME:
			value = name;
			// go through the name to find space
			valueData = value.c_str();
			while (*valueData != TEXT('\0')) {
				if (*valueData == TEXT(' ')) {
					// space found, enclode name in double quotes
					value = tstring("\"") + value + tstring("\"");
					break;
				}
				valueData++;
			}
            break;
        case CFormatField::SIZE_BYTES:
            value = bytes(size);
            break;
        case CFormatField::SIZE_SEPARATED_BYTES:
            value = separate(bytes(size));
            break;
        case CFormatField::SIZE_KBYTES:
			value = kbytes(size);
            break;
        case CFormatField::SIZE_SEPARATED_KBYTES:
			value = separate(kbytes(size));
            break;
        case CFormatField::SIZE_MBYTES:
            value = mbytes(size);
            break;
        case CFormatField::SIZE_SEPARATED_MBYTES:
			value = separate(mbytes(size));
            break;
        case CFormatField::AUTOSIZE:
            value = autosize(size);
            break;
        case CFormatField::SEPARATED_AUTOSIZE:
			value = separate(autosize(size));
            break;
        case CFormatField::DESCRIPTION:
            value = desc;
            break;
        case CFormatField::MULTILINE_DESCRIPTION:
            value = makeWindow(desc, field.width, result.length() );
            break;

        }

		// additional processing for every fields except multiline description
		if (field.type != CFormatField::MULTILINE_DESCRIPTION) {
			// resize if needed
			if (field.width != 0)
				value = resize( value, field.width );

			// truncate if needed
			if (field.precision != 0) {
				value = truncate( value, field.precision );
				// check if name was inclosed in double quotes
				if (field.type == CFormatField::NAME) {
					valueData = value.c_str();
					TCHAR	firstChar = *valueData;
					if (firstChar != TEXT('\0')) {
						int		len = value.length();
						TCHAR	lastChar  = valueData[len-1];
						// possible cases we need to fix: only first or last quote present
						if (firstChar != lastChar) {
							if (firstChar == TEXT('"')) {
								if (len == 1) {
									value = "\"\"";
								}
								else {
									value = truncate( value, len - 1);
									value += TEXT("\"");
								}
							}
							else if (lastChar == TEXT('"')) {
								if (len == 1) {
									value = "\"\"";
								}
								else {
									value = truncate( value, -(len - 1) );
									value = tstring("\"") + value;
								}
							}
						}
					}
				}
			}

		}
		// change case if needed
		if (field.modificator != CFormatField::NONE)
			value = changeCase( value, field.modificator );

        result.append( value );

        it++;
    }
    return result;
}


tstring CLineFormat::bytes(filesize_t size) {

	TCHAR	buffer[64];

	_stprintf( buffer, TEXT("%I64d"), size.QuadPart );

	return buffer;
}


tstring CLineFormat::kbytes(filesize_t size) {

	TCHAR	buffer[64];

	_stprintf( buffer, TEXT("%0.3f"), ((signed __int64)size.QuadPart) / 1024.0 );

	// set index on the last symbol of formatted string
	int	i = lstrlen(buffer) - 1;

	// shift left while character is '0'
	while ((i >= 0) && (buffer[i] == TEXT('0')))
		i--;

	// if dot ('.') reached skip it too
	if ((i >= 0) && buffer[i] == TEXT('.'))
		i--;

	// truncate buffer by placing "K\0"
	i++;
	buffer[i] = TEXT('K');
	i++;
	buffer[i] = TEXT('\0');

	return buffer;
}


tstring CLineFormat::mbytes(filesize_t size) {

	TCHAR	buffer[64];

	_stprintf( buffer, TEXT("%0.3f"), (((signed __int64)size.QuadPart) / 1024.0) / 1024.0 );

	// set index on the last symbol of formatted string
	int	i = lstrlen(buffer) - 1;

	// shift left while character is '0'
	while ((i >= 0) && (buffer[i] == TEXT('0')))
		i--;

	// if dot ('.') reached skip it too
	if ((i >= 0) && buffer[i] == TEXT('.'))
		i--;

	// truncate buffer by placing "M\0"
	i++;
	buffer[i] = TEXT('M');
	i++;
	buffer[i] = TEXT('\0');

	return buffer;
}


tstring CLineFormat::autosize(filesize_t size) {

	if (size.QuadPart < 1024) {
		return bytes(size);
	}

	if (size.QuadPart < 1024*1024) {
		return kbytes(size);
	}

	return mbytes(size);
}


tstring CLineFormat::separate(const tstring& size) {

	LPCTSTR	chars = size.c_str();

	// find end of integer part
	int	i = 0;

	while (chars[i] != TEXT('\0') && isdigit(chars[i]))
		i++;

	// now i contains length of integer part, remember it
	int	len = i;

	// copy integer part to buffer in backward order and with separation
	TCHAR	buffer[64];
	int		counter;
	int		pos;

	i--;			// last integer character
	pos     = 0;	// head of buffer
	counter = 0;	// nothing copied
	while (i >= 0) {
		buffer[pos++] = chars[i--];
		counter++;
		// place separator if 3 characters copied and there are more characters
		if (counter == 3 && i >= 0) {
			buffer[pos++] = TEXT(',');
			counter = 0;
		}
	}

	// reverse buffer

	TCHAR	result[128];
	
	i = 0;
	pos--;
	while (pos >= 0) {
		result[i++] = buffer[pos--];
	}

	// append tail
	pos = len;
	while (chars[pos] != TEXT('\0')) {
		result[i++] = chars[pos++];
	}

	// place end character
	result[i] = TEXT('\0');

	return result;
}


tstring CLineFormat::resize(const tstring& value, int width) {

	int	len = value.size();

	// if value already has width it isn't needed to be resizes
	int	positiveWidth = (width < 0 ? -width : width);
	if (positiveWidth <= len)
		return value;

	// prepare resized string
	tstring	result;

	if (width > 0) {
		// align by left
		result = value;
		result.resize( positiveWidth, TEXT(' ') );
	}
	else {
		// align by right
		result.resize( positiveWidth-len, TEXT(' ') );
		result += value;
	}

	return result;
}


tstring CLineFormat::truncate(const tstring& value, int precision) {

	int	absPrecision = (precision < 0) ? -precision : precision;

	if (value.length() <= absPrecision) {
		// don't need to truncate
		return value;
	}

	tstring	result = value;

	if (precision > 0) {
		// truncate right part
		result.resize( absPrecision );
	}
	else {
		// truncate left part
		result = result.substr( result.length() - absPrecision, absPrecision );
	}

	return result;
}


tstring CLineFormat::changeCase(const tstring& value, int modificator) {

	LPTSTR	buffer = new TCHAR[value.length() + 1];
	lstrcpy( buffer, value.c_str() );

	int	i;

	switch (modificator) {
	case CFormatField::LOWERCASE:
		{
			i = 0;
			while (buffer[i] != TEXT('\0')) {
				buffer[i] = ToLower(buffer[i]);
				i++;
			}
		}
		break;
	case CFormatField::UPPERCASE:
		{
			i = 0;
			while (buffer[i] != TEXT('\0')) {
				buffer[i] = ToUpper(buffer[i]);
				i++;
			}
		}
		break;
	}

	tstring	result = buffer;
	delete [] buffer;

	return result;
}


/**
 * Makes value word-separated in a window with a specified width and offset
 * from the start of the line; used for preparing multiline descriptions
 */
tstring CLineFormat::makeWindow(const tstring& value, int width, int offset) {

	tstring	result;		// storage for the result
	int		lineLen;	// length of the current line
	int		wordLen;	// length of the current word

	LPCTSTR	startPos = value.c_str();	// start position of the analysis
	LPCTSTR	endPos;						// end position of the analysis
	LPCTSTR	forwardPos;					// forward-looking position

	// current line is empty
	lineLen = 0;
	while (*startPos != TEXT('\0')) {
		// start from start :)
		endPos = startPos;
		// go through the spaces
		while (*endPos == TEXT(' ')) {
			endPos++;
			// do not add spaces at the beginning of the line
			// only at the middle
			if (lineLen > 0) {
				lineLen++;
				if (lineLen == width) {
					// width reached
					result += value.substr( startPos - value.c_str(), endPos-startPos );
					// adjust start position
					startPos = endPos;
					// append new line if not the end of value
					if (*endPos != TEXT('\0')) {
						result += TEXT_EOL;
						result.resize( result.size() + offset, TEXT(' ') );
					}
					// reset current line length
					lineLen = 0;
				}
			}
		}
		// stop on end of value
		if (*endPos == TEXT('\0'))
			break;
		// append spaces
		if (endPos > startPos) {
			result.resize( result.size() + endPos - startPos, TEXT(' ') );
			// shift start position
			startPos = endPos;
		}
		wordLen  = 0;
		// go through the word
		do {
			wordLen++;
			if ((wordLen + lineLen) > width) {
				// when adding current word we'll exceed allowed width
				// let's check whether word is lower than width
				forwardPos = endPos+1;
				while ((*forwardPos != TEXT('\0')) && (*forwardPos != TEXT(' ')) && (wordLen <= width)) {
					forwardPos++;
					wordLen++;
				}
				// check what limit was reached first: width or end of word
				if (wordLen <= width) {
					// we should copy word to the new line
					// fill tail of the current line with spaces
					result.resize( result.size() + width-lineLen, TEXT(' ') );
					// append line separator
					result += TEXT_EOL;
					result.resize( result.size() + offset, TEXT(' ') );
					// append word
					result += value.substr( startPos - value.c_str(), wordLen );
					// shift end pos
					endPos = forwardPos;
					// adjust start pos
					startPos = endPos;
					// setup new line length
					lineLen = wordLen;
				}
				else {
					// we should separate word
					result += value.substr( startPos - value.c_str(), width - lineLen );
					result += TEXT_EOL;
					result.resize( result.size() + offset, TEXT(' ') );
					// reset line length
					lineLen = 0;
					// move endPos
					endPos = startPos + width - lineLen;
					// adjust start pos
					startPos = endPos;
				}
				// reset word length
				wordLen = 0;
				break;
			}
			endPos++;
		} while ((*endPos != TEXT('\0')) && (*endPos != TEXT(' ')));
		if (wordLen > 0) {
			result += value.substr( startPos - value.c_str(), wordLen );
			lineLen += wordLen;
		}
		startPos = endPos;
	}

	return result;
}
